#!/usr/bin/env python3

import os
import sys

# ===== CONFIG =====
HUGE_FILE_THRESHOLD = 50  # collapse folders with MORE than this many files (recursive)
# ==================

total_files = 0
total_dirs = 0
total_size = 0

def human(size):
    for unit in ['B', 'K', 'M', 'G', 'T']:
        if size < 1024:
            return f"{size:.0f}{unit}"
        size /= 1024
    return f"{size:.0f}P"

def get_dir_stats(path):
    """Count ALL files + total size recursively"""
    files = 0
    size = 0
    for root, dirs, filenames in os.walk(path):
        for f in filenames:
            files += 1
            try:
                size += os.path.getsize(os.path.join(root, f))
            except OSError:
                pass
    return files, size

def tree(path, prefix="", is_main=False):
    global total_files, total_dirs, total_size

    try:
        entries = sorted(os.listdir(path))
    except PermissionError:
        return

    dirs = []
    files = []
    for e in entries:
        full = os.path.join(path, e)
        if os.path.isdir(full):
            dirs.append(e)
        else:
            files.append(e)

    # === MAIN FOLDER LOGIC ===
    if is_main:
        subtree_files, subtree_size = get_dir_stats(path)

        # If huge, collapse completely
        if subtree_files > HUGE_FILE_THRESHOLD:
            print(f"{prefix}├── [{human(subtree_size)}] {os.path.basename(path)} [{subtree_files} files]")
            total_dirs += 1
            total_files += subtree_files
            total_size += subtree_size
            return

        # Otherwise expand normally
        print(f"{prefix}{os.path.basename(path)}/")
        total_dirs += 1

        for f in files:
            try:
                size = os.path.getsize(os.path.join(path, f))
            except OSError:
                size = 0
            print(f"{prefix}├── [{human(size)}] {f}")
            total_files += 1
            total_size += size

        for d in dirs:
            tree(os.path.join(path, d), prefix + "│   ", is_main=False)
        return  # Don't continue below for main folders

    # === SUBFOLDER LOGIC ===
    print(f"{prefix}{os.path.basename(path)}/")
    total_dirs += 1

    for f in files:
        try:
            size = os.path.getsize(os.path.join(path, f))
        except OSError:
            size = 0
        print(f"{prefix}├── [{human(size)}] {f}")
        total_files += 1
        total_size += size

    for d in dirs:
        tree(os.path.join(path, d), prefix + "│   ", is_main=False)

# ===== RUN =====
if len(sys.argv) < 2:
    print("Usage: python3 smart_tree.py <start_folder>")
    print("Example: sudo python3 smart_tree.py / > TREE.txt")
    sys.exit(1)

ROOT = sys.argv[1]

if not os.path.isdir(ROOT):
    print(f"Error: {ROOT} not found")
    sys.exit(1)

print(f"[{ROOT}]")

# Show ALL top-level entries (no limit)
for entry in sorted(os.listdir(ROOT)):
    full = os.path.join(ROOT, entry)
    if os.path.isdir(full):
        tree(full, prefix="", is_main=True)
    else:
        try:
            size = os.path.getsize(full)
        except OSError:
            size = 0
        print(f"├── [{human(size)}] {entry}")
        total_files += 1
        total_size += size

print()
print(f"{human(total_size)} used in {total_dirs} directories, {total_files} files")
